<?php

use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\ForumController as AdminForumController;
use App\Http\Controllers\Admin\TransactionController;
use App\Http\Controllers\Admin\PackageController;
use App\Http\Controllers\Admin\UserController as AdminUserController;
use App\Http\Controllers\Admin\RolePermissionController;
use App\Http\Controllers\Admin\ConfigurationController;
use App\Http\Controllers\Admin\MapEditorController;
use App\Http\Controllers\Admin\NewsController;
use App\Http\Controllers\Admin\PluginController;
use App\Http\Controllers\Admin\CouponController;
use App\Http\Controllers\Admin\ErrorLogController;
use Illuminate\Support\Facades\Route;

// Admin routes - all protected with auth and admin dashboard access
Route::middleware(['auth', 'permission:access dashboard'])->prefix('admin')->name('admin.')->group(function () {
    // Dashboard
    Route::get('/', [DashboardController::class, 'index'])
        ->middleware('permission:access dashboard')
        ->name('dashboard');

    // Forum Management
    Route::middleware('permission:view forum categories')->group(function () {
        Route::get('/forum/categories', [AdminForumController::class, 'categories'])
            ->name('forum.categories');
    });
    
    Route::middleware('permission:create forum categories')->group(function () {
        Route::get('/forum/categories/create', [AdminForumController::class, 'createCategory'])
            ->name('forum.categories.create');
        Route::post('/forum/categories', [AdminForumController::class, 'storeCategory'])
            ->name('forum.categories.store');
    });
    
    Route::middleware('permission:edit forum categories')->group(function () {
        Route::get('/forum/categories/{category}/edit', [AdminForumController::class, 'editCategory'])
            ->name('forum.categories.edit');
        Route::put('/forum/categories/{category}', [AdminForumController::class, 'updateCategory'])
            ->name('forum.categories.update');
    });
    
    Route::middleware('permission:delete forum categories')->group(function () {
        Route::delete('/forum/categories/{category}', [AdminForumController::class, 'destroyCategory'])
            ->name('forum.categories.destroy');
    });
    
    Route::middleware('permission:view forum boards')->group(function () {
        Route::get('/forum/boards', [AdminForumController::class, 'boards'])
            ->name('forum.boards');
    });
    
    Route::middleware('permission:create forum boards')->group(function () {
        Route::get('/forum/boards/create', [AdminForumController::class, 'createBoard'])
            ->name('forum.boards.create');
        Route::post('/forum/boards', [AdminForumController::class, 'storeBoard'])
            ->name('forum.boards.store');
    });
    
    Route::middleware('permission:edit forum boards')->group(function () {
        Route::get('/forum/boards/{board}/edit', [AdminForumController::class, 'editBoard'])
            ->name('forum.boards.edit');
        Route::put('/forum/boards/{board}', [AdminForumController::class, 'updateBoard'])
            ->name('forum.boards.update');
    });
    
    Route::middleware('permission:delete forum boards')->group(function () {
        Route::delete('/forum/boards/{board}', [AdminForumController::class, 'destroyBoard'])
            ->name('forum.boards.destroy');
    });

    Route::middleware('permission:create news')->group(function () {
        // The create route needs to be defined before the show route to avoid conflicts
        Route::get('/news/create', [NewsController::class, 'create'])
            ->name('news.create');
        Route::post('/news', [NewsController::class, 'store'])
            ->name('news.store');
    });
    
    // News Management - with requested permissions
    Route::middleware('permission:manage news')->group(function () {
        Route::get('/news', [NewsController::class, 'index'])
            ->name('news.index');
        Route::get('/news/{news}', [NewsController::class, 'show'])
            ->name('news.show');
            
        Route::get('/news/{news}/edit', [NewsController::class, 'edit'])
            ->name('news.edit');
        Route::put('/news/{news}', [NewsController::class, 'update'])
            ->name('news.update');
    });
    
    
    Route::middleware('permission:delete news')->group(function () {
        Route::delete('/news/{news}', [NewsController::class, 'destroy'])
            ->name('news.destroy');
    });
    
    // Configuration Management
    Route::middleware('permission:view configurations')->group(function () {
        Route::get('configurations', [ConfigurationController::class, 'index'])
            ->name('configurations.index');
    });
    
    Route::middleware('permission:edit configurations')->group(function () {
        Route::post('configurations/batch-update', [ConfigurationController::class, 'batchUpdate'])
            ->name('configurations.batch-update');
    });
    
    
    // Role & Permission Management
    Route::middleware('permission:view roles')->group(function () {
        Route::get('/roles', [RolePermissionController::class, 'roles'])
            ->name('roles');
    });
    
    Route::middleware('permission:create roles')->group(function () {
        Route::get('/roles/create', [RolePermissionController::class, 'createRole'])
            ->name('roles.create');
        Route::post('/roles', [RolePermissionController::class, 'storeRole'])
            ->name('roles.store');
    });
    
    Route::middleware('permission:edit roles')->group(function () {
        Route::get('/roles/{role}/edit', [RolePermissionController::class, 'editRole'])
            ->name('roles.edit');
        Route::put('/roles/{role}', [RolePermissionController::class, 'updateRole'])
            ->name('roles.update');
    });
    
    Route::middleware('permission:delete roles')->group(function () {
        Route::delete('/roles/{role}', [RolePermissionController::class, 'destroyRole'])
            ->name('roles.destroy');
    });
    
    Route::middleware('permission:view permissions')->group(function () {
        Route::get('/permissions', [RolePermissionController::class, 'permissions'])
            ->name('permissions');
    });
    
    Route::middleware('permission:create permissions')->group(function () {
        Route::get('/permissions/create', [RolePermissionController::class, 'createPermission'])
            ->name('permissions.create');
        Route::post('/permissions', [RolePermissionController::class, 'storePermission'])
            ->name('permissions.store');
    });
    
    Route::middleware('permission:edit permissions')->group(function () {
        Route::get('/permissions/{permission}/edit', [RolePermissionController::class, 'editPermission'])
            ->name('permissions.edit');
        Route::put('/permissions/{permission}', [RolePermissionController::class, 'updatePermission'])
            ->name('permissions.update');
    });
    
    Route::middleware('permission:delete permissions')->group(function () {
        Route::delete('/permissions/{permission}', [RolePermissionController::class, 'destroyPermission'])
            ->name('permissions.destroy');
    });

    // Transaction Management
    Route::middleware('permission:view transactions')->group(function () {
        Route::get('/transactions', [TransactionController::class, 'index'])
            ->name('transactions.index');
        Route::get('/transactions/{transaction}', [TransactionController::class, 'show'])
            ->name('transactions.show');
    });
    
    // Package Management

    Route::middleware('permission:create packages')->group(function () {
        Route::get('/packages/create', [PackageController::class, 'create'])
            ->name('packages.create');
        Route::post('/packages', [PackageController::class, 'store'])
            ->name('packages.store');
    });

    Route::middleware('permission:view packages')->group(function () {
        Route::get('/packages', [PackageController::class, 'index'])
            ->name('packages.index');
        Route::get('/packages/{package}', [PackageController::class, 'show'])
            ->name('packages.show');
    });
    
    Route::middleware('permission:edit packages')->group(function () {
        Route::get('/packages/{package}/edit', [PackageController::class, 'edit'])
            ->name('packages.edit');
        Route::put('/packages/{package}', [PackageController::class, 'update'])
            ->name('packages.update');
    });
    
    Route::middleware('permission:delete packages')->group(function () {
        Route::delete('/packages/{package}', [PackageController::class, 'destroy'])
            ->name('packages.destroy');
    });

    Route::get('packages/servers/manage', [PackageController::class, 'servers'])->name('packages.servers');
    Route::post('packages/servers/update', [PackageController::class, 'updateServers'])->name('packages.servers.update');

    Route::resource('coupons', CouponController::class)->except(['show']);
    
    // User Management
    Route::middleware('permission:view users')->group(function () {
        Route::get('/users', [AdminUserController::class, 'index'])
            ->name('users.index');
        Route::get('/users/{user}', [AdminUserController::class, 'show'])
            ->name('users.show');
    });
    
    Route::middleware('permission:create users')->group(function () {
        Route::get('/users/create', [AdminUserController::class, 'create'])
            ->name('users.create');
        Route::post('/users', [AdminUserController::class, 'store'])
            ->name('users.store');
    });
    
    Route::middleware('permission:edit users')->group(function () {
        Route::get('/users/{user}/edit', [AdminUserController::class, 'edit'])
            ->name('users.edit');
        Route::put('/users/{user}', [AdminUserController::class, 'update'])
            ->name('users.update');
    });
    
    Route::middleware('permission:delete users')->group(function () {
        Route::delete('/users/{user}', [AdminUserController::class, 'destroy'])
            ->name('users.destroy');
    });

    Route::middleware('permission:manage server')->group(function () {
        Route::resource('servers', \App\Http\Controllers\Admin\ServerController::class);
        Route::post('servers/{server}/reset-api-key', [\App\Http\Controllers\Admin\ServerController::class, 'resetApiKey'])->name('servers.reset-api-key');
        Route::post('servers/{server}/refresh-info', [\App\Http\Controllers\Admin\ServerController::class, 'refreshInfo'])->name('servers.refresh-info');

        Route::get('/loading-screen', [App\Http\Controllers\Admin\LoadingScreenController::class, 'index'])->name('loading-screen.index');
        Route::post('/loading-screen', [App\Http\Controllers\Admin\LoadingScreenController::class, 'update'])->name('loading-screen.update');
        Route::get('/loading-screen/preview', [App\Http\Controllers\Admin\LoadingScreenController::class, 'preview'])->name('loading-screen.preview');
        Route::delete('/loading-screen/file', [App\Http\Controllers\Admin\LoadingScreenController::class, 'deleteFile'])->name('loading-screen.delete-file');
        Route::post('/loading-screen/reset', [App\Http\Controllers\Admin\LoadingScreenController::class, 'reset'])->name('loading-screen.reset');
    });

    Route::middleware('permission:manage map')->group(function () {
        // Map editor routes
        Route::get('/map/editor', [MapEditorController::class, 'index'])->name('map.editor');
        Route::post('/map/update', [MapEditorController::class, 'updateSettings'])->name('map.update');
    });

    Route::prefix('plugins')->name('plugins.')->group(function () {
        // Main plugin dashboard
        Route::get('/', [PluginController::class, 'index'])->name('index');
        
        // Plugin installation
        Route::get('/install', [PluginController::class, 'install'])->name('install');
        Route::post('/upload', [PluginController::class, 'upload'])->name('upload');
        
        // Plugin details
        Route::get('/{plugin}', [PluginController::class, 'show'])->name('show');
        
        // Plugin actions
        Route::post('/{plugin}/activate', [PluginController::class, 'activate'])->name('activate');
        Route::post('/{plugin}/deactivate', [PluginController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{plugin}', [PluginController::class, 'uninstall'])->name('uninstall');
        
        // Plugin configuration
        Route::post('/{plugin}/config', [PluginController::class, 'updateConfig'])->name('config.update');
        Route::get('/{plugin}/config/export', [PluginController::class, 'exportConfig'])->name('config.export');
        Route::post('/{plugin}/config/import', [PluginController::class, 'importConfig'])->name('config.import');
        
        // Plugin updates
        Route::get('/updates/check', [PluginController::class, 'checkUpdates'])->name('updates.check');
        Route::post('/{plugin}/update', [PluginController::class, 'update'])->name('update');
        
        // Bulk actions
        Route::post('/bulk-action', [PluginController::class, 'bulkAction'])->name('bulk');
    });
    
    // Map locations API routes
    Route::get('/map/locations', [MapEditorController::class, 'getLocations'])->name('map.locations');
    Route::post('/map/locations', [MapEditorController::class, 'storeLocation'])->name('map.locations.store');
    Route::put('/map/locations/{id}', [MapEditorController::class, 'updateLocation'])->name('map.locations.update');
    Route::delete('/map/locations/{id}', [MapEditorController::class, 'deleteLocation'])->name('map.locations.delete');
    Route::delete('/map/locations', [MapEditorController::class, 'deleteMultipleLocations'])->name('map.locations.delete-multiple');
    Route::post('/map/locations/order', [MapEditorController::class, 'updateOrder'])->name('map.locations.order');

    Route::prefix('errors')->name('errors.')->middleware('permission:view error logs')->group(function () {
        Route::get('/', [ErrorLogController::class, 'index'])->name('index');
        Route::get('/{error}', [ErrorLogController::class, 'show'])->name('show');
        
        // Actions
        Route::post('/{error}/resolve', [ErrorLogController::class, 'resolve'])->name('resolve');
        Route::post('/{error}/send', [ErrorLogController::class, 'sendToSupport'])->name('send');
        Route::get('/{error}/download', [ErrorLogController::class, 'download'])->name('download');
        Route::delete('/{error}', [ErrorLogController::class, 'destroy'])->name('destroy');
        
        // Bulk Actions
        Route::post('/bulk/resolve', [ErrorLogController::class, 'bulkResolve'])->name('bulk-resolve');
        Route::post('/bulk/send', [ErrorLogController::class, 'bulkSendToSupport'])->name('bulk-send');
        Route::get('/bulk/download', [ErrorLogController::class, 'bulkDownload'])->name('bulk-download');
        Route::post('/bulk/delete', [ErrorLogController::class, 'bulkDelete'])->name('bulk-delete');
        
        // Cleanup Actions
        Route::post('/clear/resolved', [ErrorLogController::class, 'clearResolved'])->name('clear-resolved');
        Route::post('/clear/old', [ErrorLogController::class, 'clearOld'])->name('clear-old');
    });
});